<?php
/**
 * Core: The Init class
 *
 * @link        https://oxyprops.com
 *
 * @package     OxyProps
 * @subpackage  Core
 * @author      Cédric Bontems <cedric@thewebforge.dev>
 * @since       1.5.0
 * @copyright   Copyright (c) 2022, Cédric Bontems
 * @license     https://www.gnu.org/licenses/gpl-2.0.html  GPLv2 or later
 */

namespace OxyProps\Inc;

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * OxyProps\Inc\Init.
 *
 * This class initializes the plugin and stores instances of all classes.
 *
 * @since 1.0.0 Creation.
 * @since 1.5.0 Added getters for one or all instances.
 * @author Cédric Bontems <cedric@thewebforge.dev>
 */
final class Init {

	/**
	 * All instances.
	 *
	 * @since 1.5.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @var array $instances References to all classes instances.
	 */
	private static array $instances;

	/**
	 * Get all classes that need initialization.
	 *
	 * Description.
	 *
	 * @since 1.0.0 Creation.
	 * @since 1.5.0 List update.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @return array All classes to initialize.
	 */
	public static function get_services(): array {
		return array(
			Defaults::class,
			Database::class,
			Builder::class,
			Oxyprops::class,
			Legacy::class,
			I18n::class,
			Rest_Api::class,
			Settings_Links::class,
			Helpers::class,
			Dashboard::class,
			Enqueue::class,
			Core::class,
		);
	}

	/**
	 * Instantiates classes.
	 *
	 * Loops through the classes, initialize them,
	 * and calls the register() method if it exists
	 *
	 * @since 1.0.0 Creation.
	 * @since 1.5.0 Added public visibility check and centralized instances storage.
	 * @since 1.5.1 Fixed warning Only variables should be passed by reference directly passing classname to end function.
	 * @since 1.12.0 Added CLI class to instances.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @return void
	 */
	public static function register_services(): void {
		self::$instances = array();
		foreach ( self::get_services() as $class ) {
			$service                        = $class::get_instance();
			$raw_class_name                 = explode( '\\', get_class( $service ) );
			$class_name                     = lcfirst( end( $raw_class_name ) );
			self::$instances[ $class_name ] = $service;
			if ( method_exists( self::$instances[ $class_name ], 'register' ) ) {
				$reflection = new \ReflectionMethod( self::$instances[ $class_name ], 'register' );
				if ( $reflection->isPublic() ) {
					self::$instances[ $class_name ]->register();
				}
			}
		}
		self::$instances['cli'] = new CLI();
	}

	/**
	 * Get the instance of a class.
	 *
	 * Returns the instance of a class if it exists.
	 *
	 * @since 1.5.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @param string $class_name The instance name.
	 * @return object The requested instance.
	 */
	public static function get_instance( string $class_name = '' ): object {
		if ( isset( self::$instances[ $class_name ] ) ) {
			return self::$instances[ $class_name ];
		}
		return new \stdClass();
	}

	/**
	 * Get all classes instances.
	 *
	 * Returns an array of all existing instances.
	 *
	 * @since 1.5.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @return array All instances.
	 */
	public static function get_instances(): array {
		return self::$instances;
	}
}
